import { Pool, PoolClient, QueryConfig, QueryResult } from 'pg';
import { getApiServerUrl } from '../config/serverConfig';

interface DatabaseConfig {
  host: string;
  port: number;
  user: string;
  password: string;
  database: string;
}

type PoolOptions = Pool['options'];

let cachedConfig: DatabaseConfig | null = null;

async function fetchDatabaseConfig(forceRefresh = false): Promise<DatabaseConfig> {
  if (forceRefresh) {
    cachedConfig = null;
  }

  if (cachedConfig) {
    return cachedConfig;
  }

  try {
    const apiUrl = getApiServerUrl();
    const response = await fetch(`${apiUrl}/config`, {
      headers: {
        'Content-Type': 'application/json',
      },
    });

    if (!response.ok) {
      throw new Error(`Failed to fetch config: ${response.status}`);
    }

    const result = await response.json();

    if (!result.success || !result.data) {
      throw new Error('Invalid config response format');
    }

    const config = result.data.configs;
    cachedConfig = {
      host: config.DB_HOST || 'localhost',
      port: parseInt(config.DB_PORT || '5433', 10),
      user: config.DB_USER || 'ivs',
      password: config.DB_PASSWORD || "Exitem08**",
      database: config.DB_NAME || 'airun',
    };

    return cachedConfig;
  } catch (error) {
    console.warn('Failed to fetch centralized config, using fallback:', error);

    cachedConfig = {
      host: process.env.DB_HOST || process.env.POSTGRES_HOST || 'localhost',
      port: parseInt(process.env.DB_PORT || process.env.POSTGRES_PORT || '5433', 10),
      user: process.env.DB_USER || process.env.POSTGRES_USER || 'ivs',
      password: process.env.DB_PASSWORD || process.env.POSTGRES_PASSWORD || "Exitem08**",
      database: process.env.DB_NAME || process.env.POSTGRES_DB || 'airun',
    };

    return cachedConfig;
  }
}

class ManagedDatabasePool {
  private pool: Pool | null = null;
  private creatingPool: Promise<Pool> | null = null;
  private lastOptions: PoolOptions = {} as PoolOptions;

  async ready(): Promise<void> {
    await this.ensurePool();
  }

  async reset(): Promise<void> {
    await this.ensurePool(true);
  }

  async connect(): Promise<PoolClient> {
    try {
      const pool = await this.ensurePool();
      return await pool.connect();
    } catch (error) {
      if (this.isAuthError(error)) {
        console.warn('[DB_POOL_RETRY] Authentication failed during connect, refreshing DB credentials.');
        const pool = await this.ensurePool(true);
        return pool.connect();
      }
      throw error;
    }
  }

  async query<T = any>(queryText: string | QueryConfig<any[]>, values?: any[]): Promise<QueryResult<T>> {
    const execute = async (pool: Pool) => {
      if (typeof queryText === 'string') {
        return pool.query<T>(queryText, values);
      }
      return pool.query<T>(queryText);
    };

    try {
      const pool = await this.ensurePool();
      return await execute(pool);
    } catch (error) {
      if (this.isAuthError(error)) {
        console.warn('[DB_POOL_RETRY] Authentication failed during query, refreshing DB credentials.');
        const pool = await this.ensurePool(true);
        return execute(pool);
      }
      throw error;
    }
  }

  async end(): Promise<void> {
    await this.destroyPool();
  }

  get options(): PoolOptions {
    if (this.pool) {
      return this.pool.options;
    }
    return this.lastOptions;
  }

  invalidate(): void {
    void this.destroyPool();
  }

  private async ensurePool(forceRefresh = false): Promise<Pool> {
    if (forceRefresh) {
      await this.destroyPool();
    } else if (this.pool) {
      return this.pool;
    } else if (this.creatingPool) {
      return this.creatingPool;
    }

    const creation = this.createPool(forceRefresh);
    this.creatingPool = creation;

    try {
      const pool = await creation;
      this.pool = pool;
      this.lastOptions = pool.options;
      return pool;
    } finally {
      this.creatingPool = null;
    }
  }

  private async createPool(forceRefresh: boolean): Promise<Pool> {
    const config = await fetchDatabaseConfig(forceRefresh);
    const pool = new Pool({
      host: config.host,
      port: config.port,
      database: config.database,
      user: config.user,
      password: config.password,
      max: 20,
      idleTimeoutMillis: 30000,
      connectionTimeoutMillis: 2000,
    });

    pool.on('error', (error) => {
      console.error('[DB_POOL_ERROR]', error);
    });

    return pool;
  }

  private async destroyPool(): Promise<void> {
    if (!this.pool) {
      return;
    }

    const currentPool = this.pool;
    this.pool = null;

    try {
      currentPool.removeAllListeners();
      await currentPool.end();
    } catch (error) {
      console.warn('[DB_POOL_DESTROY_WARNING]', error);
    }
  }

  private isAuthError(error: unknown): boolean {
    if (!error || typeof error !== 'object') {
      return false;
    }
    const authError = error as { code?: string; message?: string };
    return authError.code === '28P01' || !!authError.message?.toLowerCase().includes('password authentication failed');
  }
}

let poolManager: ManagedDatabasePool | null = null;

function getPoolManager(): ManagedDatabasePool {
  if (!poolManager) {
    poolManager = new ManagedDatabasePool();
  }
  return poolManager;
}

export async function getDatabasePool(): Promise<ManagedDatabasePool> {
  const manager = getPoolManager();
  await manager.ready();
  return manager;
}

export function clearConfigCache(): void {
  cachedConfig = null;
  poolManager?.invalidate();
}

export async function getCurrentDatabaseConfig(): Promise<DatabaseConfig> {
  return fetchDatabaseConfig();
}

export type DatabasePool = ManagedDatabasePool;
