#!/usr/bin/python3

import os
import sys
import time
import datetime
import filecmp
import configparser

TIMESTAMPS = "/var/log/nguardian-system.timestamps"


class nGuardianSystem():

    def __init__(self):
        self.start_time = datetime.datetime.now()
        self.logfile = open("/var/log/nguardian-system.log", "a")
        self.time_log("nguardian system started")
        self.executed = []
        self.executed_once = []
        self.overwritten = []
        self.skipped = []
        self.edited = []
        self.minimized = []
        self.original_timestamps = {}
        self.timestamps = {}
        self.timestamps_changed = False
        self.read_timestamps()

    def time_log(self, string):
        self.log("%s - %s" % (time.strftime("%Y-%m-%d %H:%M:%S"), string))

    def log(self, string):
        self.logfile.writelines("%s\n" % string)

    def quit(self):
        stop_time = datetime.datetime.now()
        self.log("Execution time: %s" % (stop_time - self.start_time))
        self.logfile.flush()
        self.logfile.close()
        sys.exit(0)

    def read_timestamps(self):
        if os.path.exists(TIMESTAMPS):
            filehandle = open(TIMESTAMPS)
            for line in filehandle:
                line = line.strip()
                line_items = line.split()
                if len(line_items) == 2:
                    self.original_timestamps[line_items[0]] = line_items[1]
                    self.timestamps[line_items[0]] = line_items[1]

    def write_timestamps(self):
        filehandle = open(TIMESTAMPS, "w")
        for filename in sorted(self.timestamps.keys()):
            line = "%s %s\n" % (filename, self.timestamps[filename])
            filehandle.write(line)
        filehandle.close()

    def has_changed(self, filename, collection, description):
        if not os.path.exists(filename):
            return False

        timestamp = os.stat(filename).st_mtime
        if (filename not in self.original_timestamps):
            has_changed = True
        else:
            has_changed = (self.original_timestamps[filename] != str(timestamp))

        if (has_changed):
            collection.append("%s (%s)" % (filename, description))
        else:
            self.skipped.append("%s (%s)" % (filename, description))
        return has_changed

    def update_timestamp(self, filename):
        timestamp = os.stat(filename).st_mtime
        self.timestamps[filename] = timestamp
        self.timestamps_changed = True

    def replace_file(self, source, destination):
        if os.path.exists(source) and os.path.exists(destination):
            if (destination not in self.overwritten) and (destination not in self.skipped):
                if filecmp.cmp(source, destination):
                    self.skipped.append(destination)
                else:
                    self.overwritten.append(destination)
                    os.system("cp " + source + " " + destination)

    def adjust(self):
        try:
            # Read configuration
            try:
                config = configparser.RawConfigParser()
                config.read('/etc/nguardian/nguardian.conf')
                self.enabled = (config.get('global', 'enabled') == "True")
                self.minimal = (config.get('global', 'minimal') == "True")
            except:
                config = configparser.RawConfigParser()
                config.add_section('global')
                config.set('global', 'enabled', 'True')
                config.set('global', 'minimal', 'False')
                config.add_section('restore')
                with open('/etc/nguardian/nguardian.conf', 'w') as configfile:
                    config.write(configfile)
                self.enabled = True
                self.minimal = False

            # Exit if disabled
            if not self.enabled:
                self.log("Disabled - Exited")
                self.quit()

            adjustment_directory = "/etc/nguardian/adjustments/"

            # Perform file execution adjustments
            for filename in os.listdir(adjustment_directory):
                basename, extension = os.path.splitext(filename)
                if extension == ".execute":
                    full_path = os.path.join(adjustment_directory, filename)
                    os.system(full_path)
                    self.executed.append(full_path)

            # For a script that runs once and does not repeat 
            for filename in os.listdir(adjustment_directory):
                basename, extension = os.path.splitext(filename)
                if extension == ".execute-once":
                    full_path = os.path.join(adjustment_directory, filename)
                    os.system(full_path)
                    os.system("mv -f " + full_path + " " + adjustment_directory + basename + ".execute-done" )
                    self.executed_once.append(full_path)

            self.log("Executed:")
            for filename in sorted(self.executed):
                self.log("  %s" % filename)

            self.log("Executed Once:")
            for filename in sorted(self.executed_once):
                self.log("  %s" % filename)
            
            self.log("--------------------")

            if self.timestamps_changed:
                self.write_timestamps()

        except Exception as detail:
            print (detail)
            self.log(detail)

main = nGuardianSystem()
main.adjust()
main.quit()